/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor.intrusiondetection.reference
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using org.owasp.appsensor;
    using org.owasp.appsensor.intrusiondetection;

    /**
     * This class represents the collection of intrusions for a specific user.
     * These records are managed by and housed in the intrusion store.  
     * <p>
     * The essential ways of interacting with this class once it is created
     * are to add intrusions and violations as events/attacks occur and to 
     * query this class for information about intrusions and violations 
     * that have occurred. 
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since September 16, 2010
     */
    public class DefaultIntrusionRecord : IntrusionRecord
    {
        /** Constant for default intusion - all */
        public const String ALL_INTRUSIONS = "Total";

        /** User this record is for */
        private String userID;

        /** Container that holds all IntrusionExceptions thrown by this user account */
        private ICollection<AppSensorIntrusion> intrusionsCommitted = new List<AppSensorIntrusion>();

        /** Container that holds the number of violations, ie crossing of given thresholds */
        private ICollection<String> violations = new List<String>();

        /** Most recent violation */
        private String lastViolation = null;

        /** Most recent response */
        private String DEFAULT_LAST_RESPONSE_ACTION = "NONE";

        //jtm - 5/23/2011 - change to concurrent map to address http://code.google.com/p/appsensor/issues/detail?id=6
        private Dictionary<String, LinkedList<String>> lastResponseActionMap = new Dictionary<String, LinkedList<String>>();

        /**
         * Constructor to create record for specified user
         * @param userID userId to create intrusion record for
         */
        public DefaultIntrusionRecord(String userID)
        {
            this.userID = userID;
        }

        /**
         * {@inheritDoc}
         */
        public void AddIntrusionToRecord(AppSensorIntrusion aie)
        {
            intrusionsCommitted.Add(aie);
        }

        /**
         * {@inheritDoc}
         */
        public int GetNumberOfAllIntrusions()
        {
            return GetNumberOfIntrusions(DefaultIntrusionRecord.ALL_INTRUSIONS);
        }

        /**
         * {@inheritDoc}
         */
        public int GetNumberOfIntrusions(String eventCode)
        {
            int defaultValue = 0;
            int count = 0;

            //if the code is for total - just get number for all intrusions
            if ((DefaultIntrusionRecord.ALL_INTRUSIONS).Equals(eventCode))
            {
                count = intrusionsCommitted.Count;
            }
            else
            {
                //handle app sensor specific intrusions individually
                //if app-sensor exception, check event code and return all of that type
                foreach (AppSensorIntrusion asi in intrusionsCommitted)
                {
                    if (asi != null && asi.EventCode != null && eventCode != null)
                    {
                        //if we get a match increment count
                        if (eventCode.Equals(asi.EventCode, StringComparison.OrdinalIgnoreCase))
                        {
                            count++;
                        }
                    }

                }//end loop
            }
            if (count < defaultValue)
            {
                // Avoid the rare integer overflow scenario
                count = Int32.MaxValue;
            }
            return count;
        }

        /**
         * {@inheritDoc}
         */
        public String UserID
        {
            get
            {
                return this.userID;
            }
        }

        /**
         * {@inheritDoc}
         */
        public void AddViolation(String eventCode)
        {
            TimeSpan currentTime = DateUtils.CurrentTime;

            String entry = currentTime.ToString() + "," + eventCode;
            violations.Add(entry);
            lastViolation = eventCode;
        }

        /**
         * {@inheritDoc}
         */
        public String GetLastViolation()
        {
            return lastViolation;
        }

        /**
         * {@inheritDoc}
         */
        public void SetLastResponseAction(String lastResponseAction, String eventCode)
        {
            //create default list to use in case there's not one in the map
            LinkedList<String> lastResponses = new LinkedList<String>();

            if (lastResponseActionMap.ContainsKey(eventCode))
            {
                //search for the specified event code
                foreach (String ec in lastResponseActionMap.Keys)
                {
                    if (ec != null && ec.Equals(eventCode))
                    {
                        //use map list if available - already has contents
                        lastResponses = lastResponseActionMap[eventCode];
                        //update map with most recent response action
                        lastResponses.AddLast(lastResponseAction);
                    }
                }
            }
            else
            {
                //there's no existing record, starting from scratch
                //add response
                lastResponses.AddLast(lastResponseAction);
                //add to map
                if (!lastResponseActionMap.ContainsKey(eventCode)) lastResponseActionMap.Add(eventCode, lastResponses);
            }
        }

        /**
         * {@inheritDoc}
         */
        public String GetLastResponseAction(String eventCode)
        {
            //use default in case last response doesn't exist yet
            String lastResponse = DEFAULT_LAST_RESPONSE_ACTION;

            //checking if code exists in map
            if (lastResponseActionMap.ContainsKey(eventCode))
            {
                //search for the specified event code
                foreach (String ec in lastResponseActionMap.Keys)
                {
                    if (ec != null && ec.Equals(eventCode))
                    {
                        LinkedList<String> lastResponses = lastResponseActionMap[eventCode];
                        //return last response
                        lastResponse = lastResponses.Last.Value;
                    }
                }
            }
            return lastResponse;
        }

        /**
         * {@inheritDoc}
         */
        public int GetNumberOfIntrusionsInInterval(String eventCode, long intervalInSeconds)
        {
            //set the begin time for considering events to the current time minus the specified time interval in seconds
            TimeSpan beginTime = DateUtils.getTimeNumberOfSecondsAgo(intervalInSeconds);

            int count = 0;

            //loop through intrusions incrementing count as we find matches
            foreach (AppSensorIntrusion asi in intrusionsCommitted)
            {
                TimeSpan airEventTime = asi.TimeStamp;
                if (airEventTime > beginTime)
                {
                    if (eventCode.Equals(DefaultIntrusionRecord.ALL_INTRUSIONS, StringComparison.OrdinalIgnoreCase)
                        || asi.EventCode.Equals(eventCode, StringComparison.OrdinalIgnoreCase))
                    {
                        //increment if we're considering either all intrusions or if we find the specific one we're looking for
                        count++;
                    }
                }
            }
            return count;
        }

        /**
         * {@inheritDoc}
         */
        public List<AppSensorIntrusion> GetCopyIntrusionsObserved()
        {
            List<AppSensorIntrusion> asiCopy = new List<AppSensorIntrusion>();
            foreach (AppSensorIntrusion asi in intrusionsCommitted)
            {
                asiCopy.Add(asi);
            }
            return asiCopy;
        }
    }
}